unit GridBlockViewer;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Grids, ExtCtrls, StdCtrls, Menus, OpenGL, ConvLib;

type
  TGridBlockViewerForm = class(TForm)
    BlockViewerMainMenu: TMainMenu;
    ViewMenu: TMenuItem;
    ReloadMenuItem: TMenuItem;
    ShowBlockInfoMenuItem: TMenuItem;
    Tools: TMenuItem;
    procedure ReloadMenuItemClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure FormPaint(Sender: TObject);
    procedure FormMouseUp(Sender: TObject; Button: TMouseButton;
                          Shift: TShiftState; X, Y: Integer);
    procedure ShowBlockInfoMenuItemClick(Sender: TObject);
    procedure FormActivate(Sender: TObject);
  private
    { Private declarations }
    procedure InitOpenGL;
    procedure Transf;
    procedure DisplayDraw;
    procedure SelectDraw;
    procedure UpdateInfoMemo;
    procedure WMEXITSIZEMOVE(var Message: TMessage); message WM_EXITSIZEMOVE;
    procedure PaintNum(X, Y: Real; Num: PChar);
    procedure finderr;
  public
    { Public declarations }
  end;

var
  GridBlockViewerForm: TGridBlockViewerForm;

implementation
uses
  MainControl,
  MainViewer, //this unit is also referenced by MainViewer
  BlockInfoViewer; //this unit is only referenced by BlockViewer
type
  TmvEnmu=(lvMASTER, lvELEMENT, lvSUBELEMENT,objNODE, objPIECE,objNONE);  
  TViewPort=packed record
    left,bottom,width,height: GLint;
  end;
var
 //reserved for update
  viewport: TViewPort;
 //dimension
  MarginRatio: Real;
 //color
  rGridColor, rUnavailColor, rAvailColor, rSelectAvailColor: GLclampf;
  gGridColor, gUnavailColor, gAvailColor, gSelectAvailColor: GLclampf;
  bGridColor, bUnavailColor, bAvailColor, bSelectAvailColor: GLclampf;
  aGridColor, aUnavailColor, aAvailColor, aSelectAvailColor: GLclampf;
  rBackGroundColor, rSelectUnavailColor: GLclampf;
  gBackGroundColor, gSelectUnavailColor: GLclampf;
  bBackGroundColor, bSelectUnavailColor: GLclampf;
  aBackGroundColor, aSelectUnavailColor: GLclampf;
  rNodeIndexColor, rBlockIndexColor: GLclampf;
  gNodeIndexColor, gBlockIndexColor: GLclampf;
  bNodeIndexColor, bBlockIndexColor: GLclampf;
  aNodeIndexColor, aBlockIndexColor: GLclampf;
 //select buffers
  SelectBuf: array [1..32] of GLuint;
  XVertices, YVertices: array [0..5] of Real;
 //for character rendering
  xShift, yShift: Real;
  CharSize: Integer;
  CusFont: TFont;
{$R *.DFM}

procedure TGridBlockViewerForm.finderr;
var errcode: GLenum;
begin
  errcode:= glGetError;
  if errcode<>GL_NO_ERROR then begin
    ShowMessage('gl error in BlockViewer:'+gluErrorString(errcode));
  end;
end;

procedure TGridBlockViewerForm.ReloadMenuItemClick(Sender: TObject);
begin
  if not (MainControlForm.StructureLoaded) then begin
   //disable the controls which should be disabled
    GridBlockViewerForm.ViewMenu.Enabled:= False;
    BlockInfoViewerForm.BlockInfoMemo.Enabled:= False;
  end else begin
   //draw the keymap
    GridBlockViewerForm.InitOpenGL;
    glViewPort(0, 0,
               ClientWidth,
               ClientHeight);
    glRenderMode(GL_RENDER);
    glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT);
    Transf;
    DisplayDraw;
    glFlush;
   //refresh the information memo
    UpdateInfoMemo;
   //enable the controls which should be enabled
    GridBlockViewerForm.ViewMenu.Enabled:= True;
    BlockInfoViewerForm.BlockInfoMemo.Enabled:= True;
  end;{if}
end;

procedure TGridBlockViewerForm.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
  BlockViewerVisible:= False;
  MainControlForm.BlockViewerMenuItem.Checked:= False;
  GridBlockViewerForm.Visible:= False;
 //make its slave form invisible
  BlockInfoViewerForm.Visible:= False; 
end;

procedure TGridBlockViewerForm.InitOpenGL;
var
  pfd: TPixelFormatDescriptor;
  FormatIndex: integer;
begin
if OpenGLReady then begin
  wglMakeCurrent(glDC,0);
  wglDeleteContext(GLContext);
  OpenGLReady:= false;
end;
  fillchar(pfd, SizeOf(pfd),0);
  with pfd do
  begin
    nSize := sizeOf(pfd);
    nVersion := 1;
    dwFlags := PFD_DRAW_TO_WINDOW or PFD_SUPPORT_OPENGL;
    iPixelType := PFD_TYPE_RGBA;
    cColorBits := 24;
    cDepthBits := 32;
    iLayerType := PFD_MAIN_PLANE;
  end;{with}
  glDC:= getDC(GridBlockViewerForm.Handle);
  FormatIndex := ChoosePixelFormat(glDC, @pfd);
  SetPixelFormat(glDC, FormatIndex, @pfd);
  GLContext := wglCreateContext(glDC);
  wglMakeCurrent(glDC, GLContext);
  glSelectBuffer(32,@SelectBuf);
  glClearColor(rBackGroundColor, gBackGroundColor,
               bBackGroundColor, aBackGroundColor);
  glEnable(GL_DEPTH_TEST);
 //the character bitmaps
  CusFont:= TFont.Create;
  CusFont.Height:= CharSize;
  SelectObject(glDC, CusFont.Handle);
  wglUseFontBitmaps(glDC, 32, 26, 33);
  glListBase(1);
 //for debugging
  finderr;
 //other
  OpenGLReady:= True;
end;

procedure TGridBlockViewerForm.FormCreate(Sender: TObject);
begin
 //dimension
  MarginRatio:= 0.15;
  xShift:= 0.05;
  yShift:= 0.05;
  CharSize:= 6; 
 //colors
  rBackGroundColor:= 0.8;
  gBackGroundColor:= 0.8;
  bBackGroundColor:= 0.8;
  aBackGroundColor:= 1.0;      //bright gray
  rGridColor:=0.1;
  gGridColor:=0.1;
  bGridColor:=0.1;
  aGridColor:=1.0;         //almost black
  rUnavailColor:=0.8;
  gUnavailColor:=0.8;
  bUnavailColor:=0.8;
  aUnavailColor:=1.0;    //bright gray
  rAvailColor:=0.6;
  gAvailColor:=0.7;
  bAvailColor:=1.0;
  aAvailColor:=1.0;    //bright blue
  rSelectAvailColor:=0.5;
  gSelectAvailColor:=0.6;
  bSelectAvailColor:=1.0;
  aSelectAvailColor:=1.0;   //darker blue
  rSelectUnavailColor:=0.7;
  gSelectUnavailColor:=0.7;
  bSelectUnavailColor:=0.7;
  aSelectUnavailColor:=1.0; //a little bit darker gray
  rNodeIndexColor:=1.0;
  gNodeIndexColor:=1.0;
  bNodeIndexColor:=0.2;
  aNodeIndexColor:=1.0;     //yellow
  rBlockIndexColor:=0.0;
  gBlockIndexColor:=0.0;
  bBlockIndexColor:=0.0;
  aBlockIndexColor:=1.0;     //black
end;{procedure FormCreate}

procedure TGridBlockViewerForm.FormMouseUp(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var hits: Integer;
begin
  if not (MainControlForm.StructureLoaded) then begin
   //do nothing
  end else if (Button=mbLeft) then begin
   //initialize the OpenGL
    InitOpenGL;
   //decide the select block
    glRenderMode(GL_SELECT);
    glInitNames;
    glPushName(0);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity;
    glGetIntegerv(GL_VIEWPORT, @viewport);
    gluPickMatrix(X, ClientHeight-Y,
                  1, 1, @viewport);
    glOrtho(0.0-NumCol*MarginRatio, NumCol*(1+MarginRatio),
            0.0-NumRow*MarginRatio, NumRow*(1+MarginRatio),
            0.0, 2.0);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity;
    gluLookAt(0.0, 0.0, 1.0,
              0.0, 0.0, 0.0,
              0.0, 1.0, 0.0);
    SelectDraw;
    hits:= glRenderMode(GL_RENDER);
    if hits=1 then SelectedBlockIndex:= SelectBuf[4];
   //redraw the key map
    glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT);
    Transf;
    DisplayDraw;
    glFlush;
   //refresh the information memo
    UpdateInfoMemo;
   //refresh the MainViewer
    MainViewerForm.DrawScene;
  end else if (Button=mbRight) then begin
   {reserved} 
  end;{if}
end; {procedure KeyMapImageMouseUp}



procedure TGridBlockViewerForm.DisplayDraw;
var i,j,k: Integer;
begin
 //draw the avail/unavail colors of blocks and and block indices
  k:= 0;
  for i:= 1 to NumCol do begin
    for j:= 1 to NumRow do begin
      k:= k+1;
      if UBlockM[k-1].mtlIndex in [1..NumMaterials] then begin
        glColor4f(rAvailColor, gAvailColor, bAvailColor, aAvailColor);
      end else begin
        glColor4f(rUnavailColor, gUnavailColor, bUnavailColor, aUnavailColor);
      end;{if}
      glBegin(GL_POLYGON);
        glVertex3f(i-1, j-1, 0.0);
        glVertex3f(  i, j-1, 0.0);
        glVertex3f(  i,   j, 0.0);
        glVertex3f(i-1,   j, 0.0);
      glEnd;
      //block indices
      glColor(rBlockIndexColor, gBlockIndexColor,
              bBlockIndexColor, aBlockIndexColor);
      PaintNum(i-0.5, j-0.5, PChar(IntToStr(k)));
    end;{for j}
  end;{for i}
 //paint the node indices
  glColor(rNodeIndexColor, gNodeIndexColor,
          bNodeIndexColor, aNodeIndexColor);
  for i:= 1 to (NumCol+1) do begin
    for j:= 1 to (NumRow+1) do begin
      PaintNum( (i-1)+xShift, (j-1)+yShift,
                PChar(IntToStr(((i-1)*(NumRow+1)+j))) );
    end;
  end;

 //draw the grid lines
  glColor4f(rGridColor, gGridColor, bGridColor, aGridColor);
  glBegin(GL_LINES);
    //vertical lines
    for i:= 0 to NumCol do begin
      glVertex3f(i, 0.0, 0.2);
      glVertex3f(i, NumRow, 0.2);
    end;{for}
    //horizontal lines
    for j:= 0 to NumRow do begin
      glVertex3f(0.0, j, 0.2);
      glVertex3f(NumCol, j, 0.2);
    end;{for}
  glEnd;
 //draw the selection retangle of selected block
  Index1DTo2D(NumRow, SelectedBlockIndex, i, j);
  if UBlockM[SelectedBlockIndex-1].mtlIndex in [1..NumMaterials] then begin
    glColor4f(rSelectAvailColor, gSelectAvailColor,
              bSelectAvailColor, aSelectAvailColor);
  end else begin
    glColor4f(rSelectUnavailColor, gSelectUnavailColor,
              bSelectUnavailColor, aSelectUnavailColor);
  end;
  glBegin(GL_POLYGON);
    glVertex3f(i-1, j-1, 0.1);
    glVertex3f(  i, j-1, 0.1);
    glVertex3f(  i,   j, 0.1);
    glVertex3f(i-1,   j, 0.1);
  glEnd;
end;{procedure DisplayDraw}

procedure TGridBlockViewerForm.SelectDraw;
var i,j,k: Integer;
begin
 //draw the avail/unavail colors of blocks
  k:= 0;
  for i:= 1 to NumCol do begin
    for j:= 1 to NumRow do begin
      k:= k+1;
      glLoadName(k);
      if UBlockM[k-1].mtlIndex in [1..NumMaterials] then begin
        glColor4f(rAvailColor, gAvailColor, bAvailColor, aAvailColor);
      end else begin
        glColor4f(rUnavailColor, gUnavailColor, bUnavailColor, aUnavailColor);
      end;{if}
      glBegin(GL_POLYGON);
        glVertex3f(i-1, j-1, 0.0);
        glVertex3f(  i, j-1, 0.0);
        glVertex3f(  i,   j, 0.0);
        glVertex3f(i-1,   j, 0.0);
      glEnd;
    end;{for j}
  end;{for i}
end;{procedure SelectDraw}



procedure TGridBlockViewerForm.Transf;
begin
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity;
  glOrtho(0.0-NumCol*MarginRatio, NumCol*(1+MarginRatio),
          0.0-NumRow*MarginRatio, NumRow*(1+MarginRatio),
          0.0, 2.0);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity;
  gluLookAt(0.0, 0.0, 1.0,
            0.0, 0.0, 0.0,
            0.0, 1.0, 0.0);
end;{procedure Transf}

procedure TGridBlockViewerForm.UpdateInfoMemo;
begin
  GetUBlockVertexCoordV(SelectedBlockIndex, True, XVertices, YVertices); 
  with BlockInfoViewerForm.BlockInfoMemo.Lines do begin
    Clear;
    Add(Format('Block Index: %d', [SelectedBlockIndex]));
    Add('Vertices:');
    Add(Format('  (%f, %f)',
        [XVertices[0],YVertices[0]]));
    Add(Format('  (%f, %f)',
        [XVertices[1],YVertices[1]]));
    Add(Format('  (%f, %f)',
        [XVertices[2],YVertices[2]]));
    Add(Format('  (%f, %f)',
        [XVertices[3],YVertices[3]]));
    Add('Material:');
    if UBlockM[SelectedBlockIndex-1].mtlIndex in [1..NumMaterials] then begin
      Add(Format('  Elastic %f',
         [Material[UBlockM[SelectedBlockIndex-1].mtlIndex-1].Elastic]));
      Add(Format('  PossionRate %f',
         [Material[UBlockM[SelectedBlockIndex-1].mtlIndex-1].PossionRate]));
      Add(Format('  UnitWeight %f',
         [Material[UBlockM[SelectedBlockIndex-1].mtlIndex-1].UnitWeight]));
    end else begin
      Add('this block is unavailable');
    end;{if}
  end;{with}
end;

procedure TGridBlockViewerForm.FormDestroy(Sender: TObject);
begin
  if OpenGLReady then begin
    wglMakeCurrent(glDC,0);
    wglDeleteContext(GLContext);
    OpenGLReady:= false;
  end;
end;

procedure TGridBlockViewerForm.FormResize(Sender: TObject);
begin
  GridBlockViewerForm.ReloadMenuItemClick(nil);
end;

procedure TGridBlockViewerForm.FormPaint(Sender: TObject);
begin
  GridBlockViewerForm.ReloadMenuItemClick(nil);
end;

procedure TGridBlockViewerForm.ShowBlockInfoMenuItemClick(Sender: TObject);
begin
  GridBlockViewerForm.ShowBlockInfoMenuItem.Checked:=
      not GridBlockViewerForm.ShowBlockInfoMenuItem.Checked;
  BlockInfoViewerForm.Visible:=
      GridBlockViewerForm.ShowBlockInfoMenuItem.Checked;
end;

procedure TGridBlockViewerForm.WMEXITSIZEMOVE(var Message: TMessage);
begin
  BlockInfoViewerForm.Top:= GridBlockViewerForm.Top+GridBlockViewerForm.Height;
  BlockInfoViewerForm.Left:= GridBlockViewerForm.Left;
end;

procedure TGridBlockViewerForm.FormActivate(Sender: TObject);
begin
  if BlockInfoViewerForm.Visible then begin
    BlockInfoViewerForm.Visible:= True;
  end;
end;

procedure TGridBlockViewerForm.PaintNum(X, Y: Real; Num: PChar);
{assumption: 1) Num is only formed by digits '0' to '9'
             2) list base is setup properly }
var l: Integer;
begin
  l:= Length(Num);
  glRasterPos3f(X, Y, 0.3);
  glCallLists(l, GL_UNSIGNED_BYTE, Num);
end;

end.
