function RegularPts = regular_simplex(d);
%
% Generate the vertices of a regular simplex in hyperspace.
%
% **********
% * Syntax *
% **********
% RegularPts = regular_simplex(d)
%
% ***************
% * Description *
% ***************
% RegularPts=regular_simplex(d) generates the vertices of a regular simplex
% in d dimensional Euclidean space.
%
% ******************
% * Input Argument *
% ******************
% d is the number of dimensions of the Euclidean space.
%   RegularPts is a d by (d+1) matrix. Each of its column stores the
% coordinates of one vertex. The distances between any pair of vertices are
% the same.
%
% *************
% * Algorithm *
% *************
% The algorithm of this routine uses the fact that, if the edge length of a
% d-dimensional regular simplex is L, then its height is L*sqrt((d+1)/2d).
% This fact can be proved by induction. Conceptually the the algorithm
% obtains an additional vertex by adding an additional dimension to update
% a lower dimensional regular simplex vertex set to a higher dimensional
% one. This routine implements it with a faster way without accumulation of
% numerical error.
%


%%%%%%%%%%%%%%%%%%%%%%%%% Internal Comments %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Written by Sui Huang, Advanced Optimization Lab, McMaster University,
% Hamilton, Ontario, Canada.
% 
% ************************
% * Modification History *
% ************************
% July   2005: First version.
% August 2005: Added comments.
% May    2006: Added dimension error check.

if (d<=0)
    error('d must be at least 1');
end
RegularPts = zeros(d, d+1);
EdgeLength = 2;
RegularPts(1,1) = -1;
RegularPts(1,2) =  1;
for i=2:d
    H = sqrt((i+1)/(2*i)) * EdgeLength;
    RegularPts(i, i+1) = H;
    RegularPts(i, i+2:(d+1)) = H/(i+1);
end

return