function [Pts, Partition, b] = cfp_gencore_random(d)
%
% A random generator of general colorful feasibility problem core cases.
%
% **********
% * Syntax *
% **********
% [Pts, Partition, b] = cfp_gencore_random(d)
%
% ***************
% * Description *
% ***************
% [Pts,Partition,b]=cfp_gencore_random(d) randomly generates a general
% colorful feasibility problem core case.
%
% ******************
% * Input Argument *
% ******************
% The input argument d is the dimension for Euclidean space.
%
% ********************
% * Output Arguments *
% ********************
% Pts is a matrix storing the coordinates of points. Each column of Pts
% stores the coordinate of one point. The number of rows is d, which is
% the number of dimensions of the Euclidean space. The points in each color
% of Pts must contain b in their convex hulls. All the colorful points are
% unit vectors. Scaling the colorful points does not affect whether a
% convex hull generated by them contain the origin, and b is the origin.
% Hence, all the colorful points can be scaled to unit vectors.
%   b is a column vector representing a point, and it is always generated
% as the origin. This output argument is for future potential update.
%   Partition is a row vector of length (d+1). Each element is an
% integer no less than (d+1), specifying the number of points in a color.
% For example [3 4 3] tells that the first 3 points in Pts are in the first
% color, the following 4 points are in the second color, and so on. This
% function only assigns (d+1) points to each color, since in practice a
% subset of size (d+1) containing b in the convex hull can always be found
% when there are more than (d+1) points. Partition is an output
% argument for future potential update.
%
% *************
% * Algorithm *
% *************
% For each color, d points on the hypersphere are selected randomly with 
% the uniform distribution on the surface of hypersphere, then the (d+1)th
% point is generated as a random convex combination of the antipodes of the
% first d points.
%   The method of generating a random point with the uniform distribution
% on the surface of hypersphere is: generate a point whose coordinates are
% random real numbers from normal distribution (mean is 0), then normalize
% the point. The reason to choose this method is because MATLAB randn()
% function to generate real number with normal distribution. This method
% can be referred to reference [1].
%
% **************
% * References *
% **************
% [1] Muller, M. E. "A Note on a Method for Generating Points Uniformly on
%     N-Dimensional Spheres" Comm. Assoc. Comput. Mach. 2, 19-20, 1959.
%

%%%%%%%%%%%%%%%%%%%%%%%%% Internal Comments %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Written by Sui Huang, Advanced Optimization Lab, McMaster University,
% Hamilton, Ontario, Canada.
% 
% ************************
% * Modification History *
% ************************
% May    2005: First version.
% August 2005: Added comments.
% May    2006: Added CoreViolate parameter.
% April  2007: Removed CoreViolate parameter.
%              Renamed to cfp_gencore_random().

%% Allocate memory for Pts.
Pts = zeros(d, (d+1)*(d+1));

%% Generate the points of each color.
for base = 0:(d+1):(d*d+d)
    % Generate a random place to place a special point in the set.
    i = rand_int(1,1,[1 (d+1)]);
    % Generate d points on the surface of hypersphere with uniform
    % distribution, by using real numbers from normal distribution.
    L = [((base+1):(base+i-1))  ((base+i+1):(base+d+1))];
    Pts(:, L) = randn(d,d);
    for j = L
        Pts(:,j) = Pts(:,j)/norm(Pts(:,j),2);
    end
    % Generate the last point of this color as a convex combination of the
    % antipodes of the first d points.
    Pts(:, (base+i)) = - Pts(:, L)* rand_real(d,1,[0 1]);
    Pts(:, (base+i)) = Pts(:, (base+i)) / norm(Pts(:, (base+i)),2);
end

%% Generate b as the origin.
b = zeros(d,1);

%% Partition tells that each color has (d+1) points.
Partition = (d+1)*ones(1,(d+1));

return
